// server.js
const express = require('express');
const axios = require('axios');
const path = require('path');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 3000;

// Serve static files from /public
app.use(express.static(path.join(__dirname, 'public')));

// Simple API route for weather: /api/weather?city=Toronto
app.get('/api/weather', async (req, res) => {
  const city = req.query.city;

  if (!city || city.trim() === '') {
    return res.status(400).json({ error: 'City name is required.' });
  }

  try {
    const apiKey = process.env.WEATHER_API_KEY;
    const url = 'https://api.openweathermap.org/data/2.5/weather';

    const response = await axios.get(url, {
      params: {
        q: city,
        appid: apiKey,
        units: 'metric', // Celsius
      },
    });

    const data = response.data;
    const result = {
      city: data.name,
      country: data.sys.country,
      temp: data.main.temp,
      feels_like: data.main.feels_like,
      description: data.weather[0].description,
      icon: data.weather[0].icon,
      humidity: data.main.humidity,
      wind_speed: data.wind.speed,
    };

    res.json(result);
  } catch (err) {
    console.error(err.response?.data || err.message);

    if (err.response && err.response.status === 404) {
      return res.status(404).json({ error: 'City not found. Please try again.' });
    }

    res.status(500).json({ error: 'An error occurred while fetching weather data.' });
  }
});

app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});
