// script.js

const form = document.getElementById('weather-form');
const cityInput = document.getElementById('city');
const errorEl = document.getElementById('error-message');

const resultSection = document.getElementById('weather-result');
const locationEl = document.getElementById('location');
const iconEl = document.getElementById('weather-icon');
const tempEl = document.getElementById('temperature');
const feelsLikeEl = document.getElementById('feels-like');
const descEl = document.getElementById('description');
const humidityEl = document.getElementById('humidity');
const windEl = document.getElementById('wind');

form.addEventListener('submit', async (e) => {
  e.preventDefault();
  const city = cityInput.value.trim();

  if (city === '') {
    showError('Please enter a city name.');
    return;
  }

  showError('');
  showLoading(true);

  try {
    const response = await fetch(`/api/weather?city=${encodeURIComponent(city)}`);

    if (!response.ok) {
      const errData = await response.json();
      throw new Error(errData.error || 'Failed to fetch weather data.');
    }

    const data = await response.json();
    displayWeather(data);
  } catch (err) {
    console.error(err);
    showError(err.message);
    resultSection.classList.add('hidden');
  } finally {
    showLoading(false);
  }
});

function displayWeather(data) {
  locationEl.textContent = `${data.city}, ${data.country}`;
  tempEl.textContent = `${data.temp.toFixed(1)}°C`;
  feelsLikeEl.textContent = `Feels like: ${data.feels_like.toFixed(1)}°C`;
  descEl.textContent = data.description.charAt(0).toUpperCase() + data.description.slice(1);
  humidityEl.textContent = `Humidity: ${data.humidity}%`;
  windEl.textContent = `Wind Speed: ${data.wind_speed} m/s`;

  iconEl.src = `https://openweathermap.org/img/wn/${data.icon}@2x.png`;
  iconEl.alt = data.description;

  resultSection.classList.remove('hidden');
}

function showError(message) {
  errorEl.textContent = message;
}

function showLoading(isLoading) {
  if (isLoading) {
    showError('Loading weather data...');
  } else if (errorEl.textContent === 'Loading weather data...') {
    showError('');
  }
}
