const express = require("express");
const path = require("path");
const tasksDb = require("./tasks");

const app = express();

// Allows Express to read JSON from requests
app.use(express.json());

// Serve frontend files
app.use(express.static(path.join(__dirname, "public")));

// GET all tasks
app.get("/tasks", (req, res) => {
    res.json(tasksDb.getAllTasks());
});

// GET one task
app.get("/tasks/:id", (req, res) => {
    const id = parseInt(req.params.id);
    const task = tasksDb.getTaskById(id);

    if (!task) {
        return res.status(404).json({ message: "Task not found" });
    }

    res.json(task);
});

// CREATE new task
app.post("/tasks", (req, res) => {
    const { title, description, status, dueDate } = req.body;

    if (!title || !description) {
        return res.status(400).json({ message: "Title and description required" });
    }

    const task = tasksDb.createTask({ title, description, status, dueDate });
    res.json({ message: "Task created", task });
});

// UPDATE task
app.put("/tasks/:id", (req, res) => {
    const id = parseInt(req.params.id);
    const { title, description, status, dueDate } = req.body;

    if (!title || !description) {
        return res.status(400).json({ message: "Title and description required" });
    }

    const updated = tasksDb.updateTask(id, { title, description, status, dueDate });

    if (!updated) {
        return res.status(404).json({ message: "Task not found" });
    }

    res.json({ message: "Task updated", task: updated });
});

// DELETE task
app.delete("/tasks/:id", (req, res) => {
    const id = parseInt(req.params.id);

    const deleted = tasksDb.deleteTask(id);
    if (!deleted) {
        return res.status(404).json({ message: "Task not found" });
    }

    res.json({ message: "Task deleted" });
});

// Start server
const PORT = 3000;
app.listen(PORT, () => console.log(`Server running at http://localhost:${PORT}`));
